﻿using System;
using System.Collections.Generic;
using System.IO.Ports;
using System.Linq;
using System.Reflection;
using System.Text;
using System.Threading;
using System.Threading.Tasks;
using AZVIC.Ei8htPOS.Utilities;
using System.ComponentModel;
using System.Data;

namespace AZVIC.Ei8htPOS.Entities
{
    public class OCBCHelper
    {
        public enum TransmissionType { Text, Hex }

        public enum MessageType { Incoming, Outgoing, Normal, Warning, Error };

        public enum TwoStepProtocol
        {
            [StringValue("00")]
            Timeout = 0x0,
            [StringValue("02")]
            StartOfText = 0x02,
            [StringValue("03")]
            EndOfText = 0x03,
            [StringValue("04")]
            EndOfTransmission = 0x04,
            [StringValue("05")]
            Enquiry = 0x05,
            [StringValue("06")]
            Acknowledge = 0x06,
            [StringValue("15")]
            NegativeAcknowledge = 0x15,
        }

        public enum OCBCReponseCode
        {
            [StringValue("The merchant must call the authorization center due to some reason.")]
            RC01,
            [StringValue("The merchant should call the authorization center, and request for a VOICE authorization with the reference as displayed(xxxxxx).")]
            RC02,
            [StringValue("Invalid Merchant.")]
            RC03,
            [StringValue("Pick up the card.")]
            RC04,
            [StringValue("The transaction was decline.")]
            RC05,
            [StringValue("Pick up the fraud card.")]
            RC07,
            [StringValue("Invalid transaction.")]
            RC12,
            [StringValue("Invalid amount.")]
            RC13,
            [StringValue("Invalid card number, re-enter the transaction.")]
            RC14,
            [StringValue("Re-enter the transaction.")]
            RC19,
            [StringValue("Unable to locate record on file.")]
            RC25,
            [StringValue("Format error.")]
            RC30,
            [StringValue("Bank not supported by switch.")]
            RC31,
            [StringValue("Expired card, declined.")]
            RC33,
            [StringValue("Allowable PIN tries exceeded.")]
            RC38,
            [StringValue("Credit account not found.")]
            RC39,
            [StringValue("Call authorization center, lost card.")]
            RC41,
            [StringValue("Call authorization center, stolen card.")]
            RC43,
            [StringValue("Transaction declined.")]
            RC51,
            [StringValue("Current account not available.")]
            RC52,
            [StringValue("Saving account not available.")]
            RC53,
            [StringValue("Card expired.")]
            RC54,
            [StringValue("PIN entered not valid.")]
            RC55,
            [StringValue("Transaction not permitted to card holder.")]
            RC57,
            [StringValue("Transaction not permitted to terminal.")]
            RC58,
            [StringValue("Exceeds withdrawal amount limit.")]
            RC61,
            [StringValue("Restricted Card.")]
            RC62,
            [StringValue("Security violation.")]
            RC63,
            [StringValue("Exceeds withdrawal frequency limit.")]
            RC65,
            [StringValue("Terminal response is too late.")]
            RC68,
            [StringValue("Allowable number of PIN entries exceeded.")]
            RC75,
            [StringValue("Descriptor error.")]
            RC76,
            [StringValue("Reconcile error.")]
            RC77,
            [StringValue("Trace # is not match.")]
            RC78,
            [StringValue("Batch number not found.")]
            RC80,
            [StringValue("Batch not found.")]
            RC85,
            [StringValue("Terminal ID cannot be found.")]
            RC89,
            [StringValue("Issuer or switch inoperative.")]
            RC91,
            [StringValue("Duplicate transmission.")]
            RC94,
            [StringValue("Reconcile error, Batch upload started.")]
            RC95,
            [StringValue("System malfunction.")]
            RC96
        }

        public enum TransactionType
        {
            [StringValue("30")]
            SaleFullPayment = 0x30,
            [StringValue("31")]
            Instalment = 0x31,
            [StringValue("32")]
            OfflineSaleFullpayment = 0x32,
            [StringValue("33")]
            payWaveSale = 0x33,
            [StringValue("34")]
            VOID = 0x34,
            [StringValue("35")]
            REFUND = 0x35,
            [StringValue("40")]
            CUPSale = 0x40,
            [StringValue("44")]
            CUPVOID = 0x44,
            [StringValue("45")]
            CUPREFUND = 0x45,
            [StringValue("01 94")]
            Success,
            [StringValue("00 39")]
            Failure,
            Error,
            [StringValue("00")]
            Approved,
            Declined
        }

        public enum AdditionalInformation
        {
            [StringValue("01")]
            Currency = 0,
            [StringValue("V16")]
            MessageVersion = 1,
            [StringValue("P")]
            SenderIndicator = 2
        }

        public class OCBCTransactionRequest
        {
            public string DeviceRequest { get; set; }
            public MessageType MessageType { get; set; }
            public TransactionType TransactionType { get; set; }
            public string ReceiptNumber { get; set; }
            public decimal Amount { get; set; }
            public int TerminalRefNo { get; set; }
            public string Currency { get; set; }

            public override string ToString()
            {
                StringBuilder request = new StringBuilder();

                //Data lenth in BCD, Length 2 byte
                request.Append("00 ");
                request.Append("43 ");

                //Sender Indicator P is default
                request.Append(TwowayConvertionHEX.ToHexString("P"));

                //Transaction Type Length is 1 byte
                request.Append(StringValueAttribute.GetStringValue(TransactionType)).Append(" ");

                //Message Version Length is 3 byte
                request.Append(TwowayConvertionHEX.ToHexString(StringValueAttribute.GetStringValue(AdditionalInformation.MessageVersion)));

                //Receipt Reference No. Length is 20
                request.Append(TwowayConvertionHEX.ToHexString(ReceiptNumber.PadRight(20, ' ')));

                //Amount Length is 12 byte. Format is without decimal 10.02 = 1002
                request.Append(TwowayConvertionHEX.ToHexString((Amount * 100).ToString("#").PadLeft(12, '0')));

                //Terminal Reference No length is 6 byte
                request.Append(TwowayConvertionHEX.ToHexString(TerminalRefNo.ToString().PadLeft(6, '0')));

                //End of the Text (ETX) Length 1 byte
                request.Append(StringValueAttribute.GetStringValue(TwoStepProtocol.EndOfText)).Append(" ");

                // LRC Checksum
                request.Append(TerminalRequest.CalculateLongitudinalRedundancyCheck(request.ToString()).ToString().PadLeft(2, '0'));

                //Start of the Text (STX) Length 1 byte
                request.Insert(0, StringValueAttribute.GetStringValue(TwoStepProtocol.StartOfText) + " ");

                //result.Append(TerminalRequest.CalculateLongitudinalRedundancyCheck(result.ToString()));

                DeviceRequest = request.ToString();

                return DeviceRequest;
            }
        }

        public class OCBCTransactionResponse : BusinessBaseClass
        {
            public OCBCTransactionResponse()
            {
            }

            public OCBCTransactionResponse(IDataReader myDataRecord)
            {
                UniqueID = myDataRecord.IsDBNull(myDataRecord.GetOrdinal("OrdersPaymentOCBCID")) ? Guid.Empty : myDataRecord.GetGuid(myDataRecord.GetOrdinal("OrdersPaymentOCBCID"));
                OrdersPaymentID = myDataRecord.IsDBNull(myDataRecord.GetOrdinal("OrdersPaymentID")) ? Guid.Empty : myDataRecord.GetGuid(myDataRecord.GetOrdinal("OrdersPaymentID"));
                DeviceRequest = myDataRecord.IsDBNull(myDataRecord.GetOrdinal("DeviceRequest")) ? string.Empty : myDataRecord.GetString(myDataRecord.GetOrdinal("DeviceRequest"));
                DeviceResponse = myDataRecord.IsDBNull(myDataRecord.GetOrdinal("DeviceResponse")) ? string.Empty : myDataRecord.GetString(myDataRecord.GetOrdinal("DeviceResponse"));
                SenderIndicator = myDataRecord.IsDBNull(myDataRecord.GetOrdinal("SenderIndicator")) ? string.Empty : myDataRecord.GetString(myDataRecord.GetOrdinal("SenderIndicator"));
                TransactionType = (TransactionType)Enum.Parse(typeof(TransactionType), myDataRecord.IsDBNull(myDataRecord.GetOrdinal("TransactionType")) ? string.Empty : myDataRecord.GetString(myDataRecord.GetOrdinal("TransactionType")));
                MessageVersion = myDataRecord.IsDBNull(myDataRecord.GetOrdinal("MessageVersion")) ? string.Empty : myDataRecord.GetString(myDataRecord.GetOrdinal("MessageVersion"));
                ECRReferenceNo = myDataRecord.IsDBNull(myDataRecord.GetOrdinal("ECRReferenceNo")) ? string.Empty : myDataRecord.GetString(myDataRecord.GetOrdinal("ECRReferenceNo"));
                Amount = myDataRecord.IsDBNull(myDataRecord.GetOrdinal("Amount")) ? 0 : myDataRecord.GetDecimal(myDataRecord.GetOrdinal("Amount"));
                ResponseCode = myDataRecord.IsDBNull(myDataRecord.GetOrdinal("ResponseCode")) ? string.Empty : myDataRecord.GetString(myDataRecord.GetOrdinal("ResponseCode"));
                MerchantID = myDataRecord.IsDBNull(myDataRecord.GetOrdinal("MerchantID")) ? string.Empty : myDataRecord.GetString(myDataRecord.GetOrdinal("MerchantID"));
                TerminalID = myDataRecord.IsDBNull(myDataRecord.GetOrdinal("TerminalID")) ? string.Empty : myDataRecord.GetString(myDataRecord.GetOrdinal("TerminalID"));
                PrimaryAccountNumber = myDataRecord.IsDBNull(myDataRecord.GetOrdinal("PrimaryAccountNumber")) ? string.Empty : myDataRecord.GetString(myDataRecord.GetOrdinal("PrimaryAccountNumber"));
                CardExpiryDate = myDataRecord.IsDBNull(myDataRecord.GetOrdinal("CardExpiryDate")) ? string.Empty : myDataRecord.GetString(myDataRecord.GetOrdinal("CardExpiryDate"));
                ApprovalCode = myDataRecord.IsDBNull(myDataRecord.GetOrdinal("ApprovalCode")) ? string.Empty : myDataRecord.GetString(myDataRecord.GetOrdinal("ApprovalCode"));
                CardLabel = myDataRecord.IsDBNull(myDataRecord.GetOrdinal("CardLabel")) ? string.Empty : myDataRecord.GetString(myDataRecord.GetOrdinal("CardLabel"));
                RRN = myDataRecord.IsDBNull(myDataRecord.GetOrdinal("RRN")) ? string.Empty : myDataRecord.GetString(myDataRecord.GetOrdinal("RRN"));
                TransactionDateTime = myDataRecord.IsDBNull(myDataRecord.GetOrdinal("TransactionDateTime")) ? DateTime.MinValue : myDataRecord.GetDateTime(myDataRecord.GetOrdinal("TransactionDateTime"));
                BatchNo = myDataRecord.IsDBNull(myDataRecord.GetOrdinal("BatchNo")) ? string.Empty : myDataRecord.GetString(myDataRecord.GetOrdinal("BatchNo"));
                CardType = myDataRecord.IsDBNull(myDataRecord.GetOrdinal("CardType")) ? string.Empty : myDataRecord.GetString(myDataRecord.GetOrdinal("CardType"));
                PersonsName = myDataRecord.IsDBNull(myDataRecord.GetOrdinal("PersonsName")) ? string.Empty : myDataRecord.GetString(myDataRecord.GetOrdinal("PersonsName"));
                TerminalRefNo = myDataRecord.IsDBNull(myDataRecord.GetOrdinal("TerminalRefNo")) ? string.Empty : myDataRecord.GetString(myDataRecord.GetOrdinal("TerminalRefNo"));
                RedemptionAmount = myDataRecord.IsDBNull(myDataRecord.GetOrdinal("RedemptionAmount")) ? 0 : myDataRecord.GetDecimal(myDataRecord.GetOrdinal("RedemptionAmount"));
                NetAmount = myDataRecord.IsDBNull(myDataRecord.GetOrdinal("NetAmount")) ? 0 : myDataRecord.GetDecimal(myDataRecord.GetOrdinal("NetAmount"));
                SequenceNo = myDataRecord.IsDBNull(myDataRecord.GetOrdinal("SequenceNo")) ? string.Empty : myDataRecord.GetString(myDataRecord.GetOrdinal("SequenceNo"));
                ResponseType = (TransactionType)Enum.Parse(typeof(TransactionType), myDataRecord.IsDBNull(myDataRecord.GetOrdinal("ResponseType")) ? string.Empty : myDataRecord.GetString(myDataRecord.GetOrdinal("ResponseType")));
                IsValidLRC = myDataRecord.IsDBNull(myDataRecord.GetOrdinal("IsValidLRC")) ? false : myDataRecord.GetBoolean(myDataRecord.GetOrdinal("IsValidLRC"));
                AdminUserAccountID = myDataRecord.IsDBNull(myDataRecord.GetOrdinal("CreatedBy")) ? Guid.Empty : myDataRecord.GetGuid(myDataRecord.GetOrdinal("CreatedBy"));
                CreatedDate = myDataRecord.IsDBNull(myDataRecord.GetOrdinal("CreatedDate")) ? DateTime.MinValue : myDataRecord.GetDateTime(myDataRecord.GetOrdinal("CreatedDate"));
                ModifiedDate = myDataRecord.IsDBNull(myDataRecord.GetOrdinal("ModifiedDate")) ? DateTime.MinValue : myDataRecord.GetDateTime(myDataRecord.GetOrdinal("ModifiedDate"));
                IsActive = myDataRecord.IsDBNull(myDataRecord.GetOrdinal("IsActive")) ? false : myDataRecord.GetBoolean(myDataRecord.GetOrdinal("IsActive"));

            }

            [DataObjectField(true, false, false)]
            public override Guid UniqueID { get; set; }

            public Guid OrdersPaymentID { get; set; }

            public string DeviceRequest { get; set; }

            public string DeviceResponse { get; set; }

            public string SenderIndicator { get; set; }

            public TransactionType TransactionType { get; set; }

            public string MessageVersion { get; set; }

            public string ECRReferenceNo { get; set; }

            public decimal Amount { get; set; }

            public string ResponseCode { get; set; }

            public string MerchantID { get; set; }

            public string TerminalID { get; set; }

            public string PrimaryAccountNumber { get; set; }

            public string CardExpiryDate { get; set; }

            public string ApprovalCode { get; set; }

            public string CardLabel { get; set; }

            public string RRN { get; set; }

            public DateTime TransactionDateTime { get; set; }

            public string BatchNo { get; set; }

            public string CardType { get; set; }

            public string PersonsName { get; set; }

            public string TerminalRefNo { get; set; }

            public decimal RedemptionAmount { get; set; }

            public decimal NetAmount { get; set; }

            public string SequenceNo { get; set; }

            public TransactionType ResponseType { get; set; }

            public bool IsValidLRC { get; set; }

            public Guid AdminUserAccountID { get; set; }

            public override DateTime CreatedDate { get; set; }

            public override DateTime ModifiedDate { get; set; }

            public override GlobalConfigurations.Mode Mode { get; set; }

            public override bool IsActive { get; set; }

            public bool VerifyLRC()
            {
                //Remove STX
                string decryptMessage = DeviceResponse.Trim().Replace(StringValueAttribute.GetStringValue(TwoStepProtocol.StartOfText), string.Empty).Trim();

                //Getting LRC 
                string ResponseLRC = decryptMessage.Trim().Substring(decryptMessage.Length - 2, 2).Trim();

                //Remove LRC
                decryptMessage = decryptMessage.Trim().Substring(0, decryptMessage.Length - 3).Trim();

                string CalculatedLRC = TerminalRequest.CalculateLongitudinalRedundancyCheck(decryptMessage);

                if (ResponseLRC.Equals(CalculatedLRC))
                    IsValidLRC = true;
                else
                {
                    ResponseType = TransactionType.Error;
                    IsValidLRC = false;
                }

                return IsValidLRC;
            }

            public void ProcessResponseString()
            {
                if (!string.IsNullOrEmpty(DeviceResponse))
                {
                    //Remove STX and ETX
                    string decryptMessage = DeviceResponse.Trim().Replace(StringValueAttribute.GetStringValue(TwoStepProtocol.StartOfText), string.Empty).Replace(StringValueAttribute.GetStringValue(TwoStepProtocol.EndOfText), string.Empty);
                    //Remove LRC 
                    decryptMessage = decryptMessage.Trim().Substring(0, decryptMessage.Length - 3).Trim();

                    //Get Message Length
                    string msgLength = decryptMessage.Substring(0, 5);

                    //Remove the Message Length
                    decryptMessage = decryptMessage.Replace(msgLength, string.Empty).Trim();

                    string[] splitMessage = decryptMessage.Replace("  ", " ").Split(new char[] { ' ' });

                    int index = 0;
                    int length = 0;
                    int tempInt = 0;
                    decimal tempDec = 0;
                    string tempString = string.Empty;

                    //For Success Result
                    if (StringValueAttribute.GetStringValue(TransactionType.Success).Equals(msgLength))
                    {
                        ResponseType = TransactionType.Success;

                        //Get Sender Indicator Length is 1 byte
                        #region Sender Indicator
                        index = 0;
                        length = 0;
                        while (index <= length)
                        {
                            SenderIndicator += TwowayConvertionHEX.FromHexString(splitMessage[index]);
                            index++;
                        }
                        #endregion

                        //Get Transaction Type Length is 1 byte
                        #region Transaction Type
                        length += 1;
                        switch (splitMessage[index++])
                        {
                            case "30":
                                TransactionType = TransactionType.SaleFullPayment;
                                break;
                            case "31":
                                TransactionType = TransactionType.Instalment;
                                break;
                            case "32":
                                TransactionType = TransactionType.OfflineSaleFullpayment;
                                break;
                            case "33":
                                TransactionType = TransactionType.payWaveSale;
                                break;
                            case "34":
                                TransactionType = TransactionType.VOID;
                                break;
                            case "35":
                                TransactionType = TransactionType.REFUND;
                                break;
                            case "40":
                                TransactionType = TransactionType.CUPSale;
                                break;
                            case "44":
                                TransactionType = TransactionType.CUPVOID;
                                break;
                            case "45":
                                TransactionType = TransactionType.CUPREFUND;
                                break;
                        }
                        #endregion

                        //Get Message Version Length is 3 byte
                        #region Message Version
                        length += 3;
                        tempString = string.Empty;
                        while (index <= length)
                        {
                            tempString += TwowayConvertionHEX.FromHexString(splitMessage[index]);
                            index++;
                        }
                        MessageVersion = tempString;
                        #endregion

                        //Get ECR Reference No Length is 20 byte
                        #region ECR Reference No
                        length += 20;
                        tempString = string.Empty;
                        while (index <= length)
                        {
                            tempString += TwowayConvertionHEX.FromHexString(splitMessage[index]);
                            index++;
                        }
                        ECRReferenceNo = tempString.Trim();
                        #endregion

                        //Get Amount Length is 12 byte
                        #region Amount
                        length += 12;
                        tempString = string.Empty;
                        while (index <= length)
                        {
                            tempString += TwowayConvertionHEX.FromHexString(splitMessage[index]);
                            index++;
                        }
                        if (decimal.TryParse(tempString, out tempDec))
                            Amount = tempDec / 100;
                        else
                            Amount = 0;
                        #endregion

                        //Get Response Code Length is 2 byte
                        #region Response Code
                        length += 2;
                        tempString = string.Empty;
                        while (index <= length)
                        {
                            tempString += TwowayConvertionHEX.FromHexString(splitMessage[index]);
                            index++;
                        }
                        ResponseCode = tempString;
                        #endregion

                        //Get Merchant ID Length is 15 byte
                        #region Merchant ID
                        length += 15;
                        tempString = string.Empty;
                        while (index <= length)
                        {
                            tempString += TwowayConvertionHEX.FromHexString(splitMessage[index]);
                            index++;
                        }
                        MerchantID = tempString;
                        #endregion

                        //Get Terminal ID Length is 8 byte
                        #region Terminal ID
                        length += 8;
                        tempString = string.Empty;
                        while (index <= length)
                        {
                            tempString += TwowayConvertionHEX.FromHexString(splitMessage[index]);
                            index++;
                        }
                        TerminalID = tempString;
                        #endregion

                        //Get PAN Length is 20 byte
                        #region PAN
                        length += 20;
                        tempString = string.Empty;
                        while (index <= length)
                        {
                            tempString += TwowayConvertionHEX.FromHexString(splitMessage[index]);
                            index++;
                        }
                        PrimaryAccountNumber = tempString;
                        #endregion

                        //Get Card Exp.Date Length is 4 byte
                        #region Card Exp.Date
                        length += 4;
                        tempString = string.Empty;
                        while (index <= length)
                        {
                            tempString += TwowayConvertionHEX.FromHexString(splitMessage[index]);
                            index++;
                        }
                        CardExpiryDate = tempString;
                        #endregion

                        //Get Approval Code Length is 6 byte
                        #region Approval Code
                        length += 6;
                        tempString = string.Empty;
                        while (index <= length)
                        {
                            tempString += TwowayConvertionHEX.FromHexString(splitMessage[index]);
                            index++;
                        }
                        ApprovalCode = tempString;
                        #endregion

                        //Get Card Label Length is 10 byte
                        #region Card Label
                        length += 10;
                        tempString = string.Empty;
                        while (index <= length)
                        {
                            tempString += TwowayConvertionHEX.FromHexString(splitMessage[index]);
                            index++;
                        }
                        CardLabel = tempString;
                        #endregion

                        //Get RRN Length is 12 byte
                        #region RRN
                        length += 12;
                        tempString = string.Empty;
                        while (index <= length)
                        {
                            tempString += TwowayConvertionHEX.FromHexString(splitMessage[index]);
                            index++;
                        }
                        RRN = tempString;
                        #endregion

                        //Get Date/Time Length is 12 byte
                        #region Date / Time
                        length += 12;
                        tempString = string.Empty;
                        while (index <= length)
                        {
                            tempString += TwowayConvertionHEX.FromHexString(splitMessage[index]);
                            index++;
                        }
                        try
                        {
                            TransactionDateTime = DateTime.ParseExact(tempString, "YYMMDDHHmmSS", null);
                        }
                        catch
                        {
                            TransactionDateTime = DateTime.MinValue;
                        }
                        #endregion

                        //Get Batch No Length is 6 byte
                        #region Batch No
                        length += 6;
                        tempString = string.Empty;
                        while (index <= length)
                        {
                            tempString += TwowayConvertionHEX.FromHexString(splitMessage[index]);
                            index++;
                        }
                        BatchNo = tempString;
                        #endregion

                        //Get Card Type Length is 2 byte
                        #region Card Type
                        length += 2;
                        tempString = string.Empty;
                        while (index <= length)
                        {
                            tempString += TwowayConvertionHEX.FromHexString(splitMessage[index]);
                            index++;
                        }
                        CardType = tempString;
                        #endregion

                        //Get Person's Name Length is 30 byte
                        #region Person's Name
                        length += 30;
                        tempString = string.Empty;
                        while (index <= length)
                        {
                            tempString += TwowayConvertionHEX.FromHexString(splitMessage[index]);
                            index++;
                        }
                        PersonsName = tempString;
                        #endregion

                        //Get Terminal Ref. No Length is 6 byte
                        #region Terminal Ref. No
                        length += 6;
                        tempString = string.Empty;
                        while (index <= length)
                        {
                            tempString += TwowayConvertionHEX.FromHexString(splitMessage[index]);
                            index++;
                        }
                        TerminalRefNo = tempString;
                        #endregion

                        //Get Redemption Amount is 12 byte
                        #region Redemption Amount
                        length += 12;
                        tempString = string.Empty;
                        while (index <= length)
                        {
                            tempString += TwowayConvertionHEX.FromHexString(splitMessage[index]);
                            index++;
                        }
                        if (decimal.TryParse(tempString, out tempDec))
                            RedemptionAmount = tempDec / 100;
                        else
                            RedemptionAmount = 0;
                        #endregion

                        //Get Net Amount is 12 byte
                        #region Net Amount
                        length += 12;
                        tempString = string.Empty;
                        while (index <= length)
                        {
                            tempString += TwowayConvertionHEX.FromHexString(splitMessage[index]);
                            index++;
                        }
                        if (decimal.TryParse(tempString, out tempDec))
                            NetAmount = tempDec / 100;
                        else
                            NetAmount = 0;
                        #endregion
                    }
                    else if (StringValueAttribute.GetStringValue(TransactionType.Failure).Equals(msgLength))
                    {
                        ResponseType = TransactionType.Failure;

                        //Get Sender Indicator Length is 1 byte
                        #region Sender Indicator
                        index = 0;
                        length = 0;
                        while (index <= length)
                        {
                            SenderIndicator += TwowayConvertionHEX.FromHexString(splitMessage[index]);
                            index++;
                        }
                        #endregion

                        //Get Transaction Type Length is 1 byte
                        #region Transaction Type
                        length += 1;
                        switch (splitMessage[index])
                        {
                            case "30":
                                TransactionType = TransactionType.SaleFullPayment;
                                break;
                            case "31":
                                TransactionType = TransactionType.Instalment;
                                break;
                            case "32":
                                TransactionType = TransactionType.OfflineSaleFullpayment;
                                break;
                            case "33":
                                TransactionType = TransactionType.payWaveSale;
                                break;
                            case "34":
                                TransactionType = TransactionType.VOID;
                                break;
                            case "35":
                                TransactionType = TransactionType.REFUND;
                                break;
                            case "40":
                                TransactionType = TransactionType.CUPSale;
                                break;
                            case "44":
                                TransactionType = TransactionType.CUPVOID;
                                break;
                            case "45":
                                TransactionType = TransactionType.CUPREFUND;
                                break;
                        }
                        index++;
                        #endregion

                        //Get Sequence No Length is 3 byte
                        #region Sequence No
                        length += 3;
                        tempString = string.Empty;
                        while (index <= length)
                        {
                            tempString += TwowayConvertionHEX.FromHexString(splitMessage[index]);
                            index++;
                        }
                        SequenceNo = tempString;
                        #endregion

                        //Get ECR Reference No Length is 20 byte
                        #region ECR Reference No
                        length += 20;
                        tempString = string.Empty;
                        while (index <= length)
                        {
                            tempString += TwowayConvertionHEX.FromHexString(splitMessage[index]);
                            index++;
                        }
                        ECRReferenceNo = tempString.Trim();
                        #endregion

                        //Get Amount Length is 12 byte
                        #region Amount
                        length += 12;
                        tempString = string.Empty;
                        while (index <= length)
                        {
                            tempString += TwowayConvertionHEX.FromHexString(splitMessage[index]);
                            index++;
                        }
                        if (decimal.TryParse(tempString, out tempDec))
                            Amount = tempDec / 100;
                        else
                            Amount = 0;
                        #endregion

                        //Get Response Code Length is 2 byte
                        #region Response Code
                        length += 2;
                        tempString = string.Empty;
                        while (index <= length)
                        {
                            tempString += TwowayConvertionHEX.FromHexString(splitMessage[index]);
                            index++;
                        }
                        ResponseCode = tempString;
                        #endregion
                    }
                    else
                    {
                        ResponseType = TransactionType.Error;
                    }
                }
            }
        }

        public static class TerminalRequest
        {
            public static string CalculateLongitudinalRedundancyCheck(string source)
            {
                int checksum = 0;
                foreach (string chData in source.Trim().Split(new char[] { ' ' }))
                {
                    checksum ^= int.Parse(chData, System.Globalization.NumberStyles.HexNumber);
                }

                return checksum.ToString("X2");
            }
        }
    }
}
